# 🔐 OTP Selling System - Complete Documentation

## 📋 Table of Contents
1. [System Overview](#system-overview)
2. [Features](#features)
3. [Installation](#installation)
4. [Usage Guide](#usage-guide)
5. [Admin Panel](#admin-panel)
6. [Link Masking Feature](#link-masking-feature)
7. [Database Schema](#database-schema)
8. [API & Functions](#api--functions)
9. [Troubleshooting](#troubleshooting)

---

## System Overview

**OTP Selling System** ek complete web-based platform hai jo OTP (One-Time Password) numbers ko sell karne ke liye banaya gaya hai. Yeh system:

- Users ko register aur login karne deta hai
- WhatsApp aur Facebook OTP numbers sell karta hai
- Admin ko bulk mein numbers upload karne ka option deta hai
- Link masking feature se original links ko hide karta hai
- Manual deposit approval system hai
- Beautiful aur responsive UI hai

### Technology Stack
- **Backend**: PHP 7.4+
- **Database**: MySQL 5.7+
- **Frontend**: HTML5, CSS3, JavaScript
- **Server**: Apache/Nginx
- **Security**: bcrypt password hashing, SQL injection prevention

---

## Features

### 👤 User Features

#### 1. Registration & Authentication
```
- Email-based registration
- WhatsApp number collection
- Secure password hashing (bcrypt)
- Session-based authentication
- Login/Logout functionality
```

#### 2. Dashboard
- Current balance display
- Available OTP numbers count
- Quick access menu
- Real-time statistics

#### 3. Get OTP (Purchase)
```
- WhatsApp numbers: Configurable price
- Facebook numbers: Configurable price
- Automatic balance deduction
- Instant OTP delivery
- Stock management
```

#### 4. Deposit System
```
- Multiple payment methods
- Screenshot upload
- Admin approval workflow
- Deposit history tracking
- Status notifications (pending/approved/rejected)
```

#### 5. Purchase History
```
- View all purchased OTPs
- Phone number display
- Service type (WhatsApp/Facebook)
- OTP link access
- Purchase date/time
```

### 🔧 Admin Features

#### 1. Dashboard
```
- Total users count
- WhatsApp numbers (total/available)
- Facebook numbers (total/available)
- Pending deposits count
- Total revenue calculation
```

#### 2. OTP Management
```
Bulk Upload:
- TXT file format: phone_number|otp_link
- Service type selection (WhatsApp/Facebook)
- Price setting
- Batch processing

Manual Add:
- Single OTP entry
- Phone number input
- OTP link input
- Service type selection
- Price configuration

List Management:
- View all OTP numbers
- Status tracking (available/sold)
- Delete functionality
- Search and filter
```

#### 3. Deposit Management
```
- View all deposit requests
- Pending deposits list
- Screenshot preview
- Approve/Reject functionality
- Balance update on approval
- Status tracking
```

#### 4. Payment Methods
```
- Add new payment methods
- Edit method details
- Delete methods
- Manage active/inactive status
- Display payment instructions
```

#### 5. User Management
```
- View all users
- User email display
- Balance tracking
- Account status
- Delete users
- Join date tracking
```

#### 6. Settings
```
- WhatsApp price configuration
- Facebook price configuration
- Real-time price updates
- Service pricing management
```

---

## Installation

### Prerequisites
- PHP 7.4 or higher
- MySQL 5.7 or higher
- Web server (Apache/Nginx)
- 100MB+ disk space
- FTP/SSH access to hosting

### Step-by-Step Installation

#### 1. Upload Files
```bash
# Via FTP:
- Connect to your hosting
- Upload all files to public_html/otp_system/
- Create 'uploads' folder

# Via SSH:
mkdir -p public_html/otp_system
cd public_html/otp_system
# Upload files here
```

#### 2. Set Permissions
```bash
chmod 755 uploads
chmod 644 config.php
chmod 644 .htaccess
```

#### 3. Run Installer
```
1. Open: http://yourdomain.com/otp_system/install.php
2. Enter database credentials
3. Create admin account
4. System will be ready to use
```

#### 4. First Login
```
Email: Your admin email
Password: Your admin password
URL: http://yourdomain.com/otp_system/login.php
```

### cPanel Installation

#### Via cPanel File Manager:
```
1. Create MySQL Database:
   - cPanel → MySQL Databases
   - Create database (e.g., otp_system)
   - Create user and assign privileges

2. Upload Files:
   - File Manager → public_html
   - Create folder 'otp_system'
   - Upload all files

3. Set Permissions:
   - Right-click 'uploads' → Permissions → 755
   - Right-click 'config.php' → Permissions → 644

4. Run Installer:
   - Browser: http://yourdomain.com/otp_system/install.php
   - Follow installation steps
```

---

## Usage Guide

### For Users

#### Registration
```
1. Click "Register here" on login page
2. Enter email address
3. Enter WhatsApp number
4. Create password (min 6 characters)
5. Confirm password
6. Click Register
7. Login with credentials
```

#### Purchasing OTP
```
1. Login to dashboard
2. Click "Get OTP"
3. Choose service (WhatsApp or Facebook)
4. Check available numbers
5. Click "Purchase Now"
6. Balance automatically deducted
7. OTP link displayed in history
```

#### Making Deposit
```
1. Go to "Deposit" section
2. Enter amount
3. Select payment method
4. Upload payment screenshot
5. Submit request
6. Wait for admin approval
7. Balance updated after approval
```

#### Viewing History
```
1. Click "History"
2. See all purchased OTPs
3. View phone numbers
4. Access OTP links
5. Check purchase dates
```

### For Admins

#### Adding OTP Numbers (Bulk)
```
1. Go to Admin Dashboard
2. Click "Manage OTP Numbers"
3. Select service type (WhatsApp/Facebook)
4. Enter price
5. Upload TXT file with format:
   phone_number|otp_link
6. System processes all entries
7. Numbers ready for sale
```

#### Adding OTP Numbers (Manual)
```
1. Go to "Manage OTP Numbers"
2. Fill in phone number
3. Enter OTP link
4. Select service type
5. Enter price
6. Click "Add OTP Number"
```

#### Managing Deposits
```
1. Go to "Manage Deposits"
2. View pending requests
3. Check screenshot
4. Click "Approve" or "Reject"
5. Balance updated on approval
6. User notified of status
```

#### Setting Prices
```
1. Go to "Settings"
2. Enter WhatsApp price
3. Enter Facebook price
4. Click "Save Settings"
5. Prices updated immediately
```

---

## Link Masking Feature

### How It Works

The system automatically masks OTP links so users see your domain instead of the original source.

**Example:**
```
Admin uploads:
  Phone: 3325292763
  Link: https://sms222.us?token=i7WGStiD6d01122113

User sees:
  3325292763|https://yourdomain.com/proxy.php?token=xxxxx

When user clicks:
  - Redirected to actual SMS link
  - Original link remains hidden
  - User only sees your domain
```

### Implementation Details

```php
// In proxy.php:
- Receives masked token
- Looks up actual link in database
- Redirects to real link
- User never sees original domain
```

### Benefits
- Branding: Users see your domain
- Security: Original source hidden
- Control: All links go through your system
- Tracking: Can monitor link usage

---

## Database Schema

### Users Table
```sql
CREATE TABLE users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    email VARCHAR(255) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    whatsapp_number VARCHAR(20),
    balance DECIMAL(10, 2) DEFAULT 0.00,
    is_admin INT DEFAULT 0,
    status INT DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);
```

### OTP Numbers Table
```sql
CREATE TABLE otp_numbers (
    id INT PRIMARY KEY AUTO_INCREMENT,
    phone_number VARCHAR(20) NOT NULL,
    otp_link TEXT NOT NULL,
    service_type ENUM('whatsapp', 'facebook') NOT NULL,
    price DECIMAL(10, 2) NOT NULL,
    country VARCHAR(50) DEFAULT 'USA',
    is_available INT DEFAULT 1,
    purchased_by INT,
    purchased_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (purchased_by) REFERENCES users(id) ON DELETE SET NULL
);
```

### Deposits Table
```sql
CREATE TABLE deposits (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    amount DECIMAL(10, 2) NOT NULL,
    payment_method VARCHAR(100),
    screenshot_path VARCHAR(255),
    status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
    admin_notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    approved_at TIMESTAMP NULL,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);
```

### Purchase History Table
```sql
CREATE TABLE purchase_history (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    otp_number_id INT NOT NULL,
    phone_number VARCHAR(20) NOT NULL,
    otp_link TEXT NOT NULL,
    service_type VARCHAR(50),
    price DECIMAL(10, 2) NOT NULL,
    purchase_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (otp_number_id) REFERENCES otp_numbers(id) ON DELETE CASCADE
);
```

### Deposit Methods Table
```sql
CREATE TABLE deposit_methods (
    id INT PRIMARY KEY AUTO_INCREMENT,
    method_name VARCHAR(100) NOT NULL,
    method_details TEXT,
    is_active INT DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
```

### Settings Table
```sql
CREATE TABLE settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(100) UNIQUE NOT NULL,
    setting_value TEXT,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);
```

---

## API & Functions

### Helper Functions (config.php)

```php
// Sanitize user input
sanitize($data)

// Redirect to URL
redirect($url)

// Check if user logged in
isLoggedIn()

// Check if user is admin
isAdmin()

// Require login (redirect if not logged in)
requireLogin()

// Require admin (redirect if not admin)
requireAdmin()
```

### Database Queries

#### Get User Balance
```php
$query = "SELECT balance FROM users WHERE id = $user_id";
$result = $conn->query($query);
$user = $result->fetch_assoc();
echo $user['balance'];
```

#### Update User Balance
```php
$new_balance = $current_balance - $amount;
$conn->query("UPDATE users SET balance = $new_balance WHERE id = $user_id");
```

#### Get Available OTP Numbers
```php
$query = "SELECT * FROM otp_numbers 
          WHERE is_available = 1 
          AND service_type = 'whatsapp'";
$result = $conn->query($query);
```

#### Record Purchase
```php
$conn->query("INSERT INTO purchase_history 
            (user_id, otp_number_id, phone_number, otp_link, service_type, price) 
            VALUES ($user_id, $otp_id, '$phone', '$link', '$service', $price)");
```

---

## Troubleshooting

### Installation Issues

#### Error: "Connection failed"
```
Solution:
1. Verify database credentials
2. Check MySQL is running
3. Ensure user has database creation privileges
4. Try with localhost instead of 127.0.0.1
```

#### Error: "Permission denied" on uploads
```
Solution:
1. SSH: chmod 755 uploads
2. cPanel: Right-click → Permissions → 755
3. Restart web server
```

#### Error: "installed.lock already exists"
```
Solution:
1. Delete installed.lock file
2. Go to install.php again
3. Complete installation
```

### Login Issues

#### Can't login after installation
```
Solution:
1. Verify admin email is correct
2. Check password (case-sensitive)
3. Clear browser cookies
4. Try different browser
5. Check database connection
```

#### Session expires too quickly
```
Solution:
1. Edit config.php
2. Increase session timeout:
   ini_set('session.gc_maxlifetime', 86400);
3. Restart web server
```

### Deposit Issues

#### Screenshots not uploading
```
Solution:
1. Check uploads folder permissions (755)
2. Verify file size < 5MB
3. Use JPG, PNG, or GIF format
4. Check disk space available
```

#### Balance not updating after approval
```
Solution:
1. Verify deposit status changed to 'approved'
2. Check user ID in database
3. Manually update balance:
   UPDATE users SET balance = balance + amount WHERE id = user_id;
```

### OTP Issues

#### Bulk upload not working
```
Solution:
1. Check TXT file format: phone|link
2. Verify no extra spaces
3. Use UTF-8 encoding
4. Check file size < 10MB
5. Ensure one entry per line
```

#### OTP numbers not appearing
```
Solution:
1. Verify numbers are marked as available
2. Check service_type matches
3. Verify price > 0
4. Check database connection
5. Refresh page
```

---

## Security Features

### Implemented Security Measures
- ✅ bcrypt password hashing
- ✅ SQL injection prevention (prepared statements)
- ✅ Session-based authentication
- ✅ Admin-only access control
- ✅ Input sanitization
- ✅ File upload validation
- ✅ CSRF protection ready
- ✅ XSS prevention

### Security Best Practices
1. Change default admin password immediately
2. Use HTTPS in production
3. Keep PHP updated
4. Regular database backups
5. Monitor file permissions
6. Use strong passwords
7. Regular security audits

---

## File Structure

```
otp_system/
├── config.php                  # Database & helper functions
├── database.sql                # Database schema
├── install.php                 # Installation wizard
├── login.php                   # User login
├── register.php                # User registration
├── proxy.php                   # Link masking handler
├── .htaccess                   # Apache configuration
├── user/
│   ├── dashboard.php           # User dashboard
│   ├── get_otp.php             # Purchase OTP
│   ├── deposit.php             # Deposit funds
│   ├── history.php             # Purchase history
│   └── logout.php              # Logout
├── admin/
│   ├── dashboard.php           # Admin dashboard
│   ├── manage_otp.php          # OTP management
│   ├── manage_deposits.php     # Deposit management
│   ├── manage_methods.php      # Payment methods
│   ├── manage_users.php        # User management
│   ├── settings.php            # System settings
│   └── logout.php              # Logout
├── uploads/                    # User uploads (deposits)
├── example_otp_numbers.txt     # Example bulk upload
├── SETUP_GUIDE.md              # Setup instructions
└── README.md                   # This file
```

---

## Support & Maintenance

### Regular Maintenance Tasks
1. Backup database weekly
2. Review pending deposits daily
3. Monitor user activity
4. Update prices as needed
5. Clean up old records
6. Check server logs

### Backup Database
```bash
mysqldump -u username -p database_name > backup.sql
```

### Restore Database
```bash
mysql -u username -p database_name < backup.sql
```

---

## Version History

- **v1.0** (2024): Initial release
  - User registration & login
  - OTP purchase system
  - Admin panel
  - Link masking
  - Deposit management

---

## License & Terms

This system is provided as-is. Use responsibly and comply with local laws regarding OTP services.

---

**Last Updated**: 2024
**Support**: For issues, check this documentation or contact your hosting provider
